<?php

namespace App\Controllers;


use CodeIgniter\Controller;
use App\Models\GenericModel;
use Config\Database;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;

class CrudController extends Controller
{
    protected $helpers = ['url', 'form', 'filesystem'];
    protected $db;

    public function __construct()
    {
        $this->db = Database::connect();
    }

    protected function newModel(string $table)
    {
        $model = new GenericModel();
        $model->setTable($table);
        return $model;
    }


    public function index($table = null)
    {
        $session = session();
        $model = new \App\Models\GenericModel();
        $db = \Config\Database::connect();

        // Ambil role user
        $userRole = $session->get('user_role');

        // Ambil tabel aktif dari table_settings
        $query = $db->table('table_settings')
            ->select('table_name, icon')
            ->where('role', $userRole)
            ->where('menu', 'aktif')
            ->get()
            ->getResultArray();

        $tables = [];
        $icons = [];
        foreach ($query as $row) {
            $tables[] = $row['table_name'];
            $icons[strtolower($row['table_name'])] = $row['icon'];
        }

        // Ambil urutan & kategori dari tabel menu
        $menuQuery = $db->table('menu')
            ->select('table_name, kategori, nomor, menu_order')
            ->whereIn('table_name', $tables)
            ->orderBy('nomor', 'ASC')
            ->orderBy('menu_order', 'ASC')
            ->get()
            ->getResultArray();

        // Grouping berdasarkan kategori
        $groupedTables = [];
        foreach ($menuQuery as $row) {
            $kategori = $row['kategori'] ?: 'Tanpa Kategori';
            $groupedTables[$kategori][] = $row['table_name'];
        }

        // === Data untuk view ===
        $data['tables'] = $tables;
        $data['icons'] = $icons;
        $data['groupedTables'] = $groupedTables; // <── penting!
        $data['table'] = $table ?? null;
        $data['fields'] = $table ? $model->getTableFields($table) : [];
        $data['meta'] = $table ? $model->getTableMeta($table) : [];
        $data['user'] = $session->get();

        if (empty($tables)) {
            $data['noAccess'] = 'Tidak ada tabel yang diizinkan untuk role ini.';
        }

        return view('crud/content', $data);
    }

    public function loadTableContent(string $table)
    {
        $model = new \App\Models\GenericModel();
        $db = \Config\Database::connect();
        $session = session();

        // Ambil role user dari session
        $userRole = $session->get('user_role') ?? $session->get('role') ?? null;

        // Ambil semua field dari tabel
        $fields = $model->getTableFields($table);
        $meta   = $model->getTableMeta($table);
        $rows   = $model->getTableData($table);

        // === Ambil pengaturan tombol (table_settings) ===
        $buttonSettings = array_fill_keys(
            ['tambah', 'print', 'export', 'import', 'view', 'edit', 'delete', 'generate', 'delete_all'],
            'nonaktif'
        );

        if ($userRole) {
            $query = $db->table('table_settings')
                ->select('tambah, print, export, import, view, edit, delete, generate, delete_all')
                ->where('role', $userRole)
                ->where('table_name', $table)
                ->get()
                ->getRowArray();

            if ($query) {
                foreach ($buttonSettings as $k => $v) {
                    if (isset($query[$k]) && $query[$k] === 'aktif') {
                        $buttonSettings[$k] = 'aktif';
                    }
                }
            }
        }

        // === Ambil field visible dari field_settings ===
        $visibleFields = [];
        if ($userRole) {
            $fieldSettings = $db->table('field_settings')
                ->select('field_name')
                ->where('table_name', $table)
                ->where('role', $userRole)
                ->where('visible', 'aktif')
                ->get()
                ->getResultArray();

            foreach ($fieldSettings as $fs) {
                $visibleFields[] = $fs['field_name'];
            }
        }

        // Jika tidak ada pengaturan kolom aktif → tampilkan semua
        if (empty($visibleFields)) {
            $visibleFields = $fields;
        }

        // === Ambil pengaturan toggle dari toggle_fields ===
        $toggleFields = [];
        if ($userRole) {
            $toggleSettings = $db->table('toggle_fields')
                ->select('field_name')
                ->where('table_name', $table)
                ->where('role', $userRole)
                ->where('toggle_on', 'aktif')
                ->get()
                ->getResultArray();

            foreach ($toggleSettings as $ts) {
                $toggleFields[] = $ts['field_name'];
            }
        }

        // === 🔹 Ambil ikon tabel dari table_settings ===
        $icon = 'fa-database'; // default
        if ($userRole) {
            $iconQuery = $db->table('table_settings')
                ->select('icon')
                ->where('role', $userRole)
                ->where('table_name', $table)
                ->get()
                ->getRowArray();

            if ($iconQuery && !empty($iconQuery['icon'])) {
                $icon = $iconQuery['icon'];
            }
        }

        // === 🔹 Ambil informasi tambahan dari tabel infotabel ===
        $infoTabel = '';
        $infoUser  = '';

        if ($userRole) {
            $infoQuery = $db->table('infotabel')
                ->select('info_tabel, info_user')
                ->where('nama_tabel', $table)
                ->where('role', $userRole)
                ->get()
                ->getRowArray();

            if ($infoQuery) {
                $infoTabel = $infoQuery['info_tabel'] ?? '';
                $infoUser  = $infoQuery['info_user'] ?? '';
            }
        }

        // === Kembalikan ke view ===
        return view('crud/table_content', [
            'table'        => $table,
            'fields'       => $visibleFields, // field yang tampil
            'meta'         => $meta,
            'rows'         => $rows,
            'buttons'      => $buttonSettings,
            'toggleFields' => $toggleFields,   // field yang toggle-nya aktif
            'icon'         => $icon,           // 🔹 ikon tabel dikirim ke view
            'infoTabel'    => $infoTabel,      // 🔹 info_tabel dari infotabel
            'infoUser'     => $infoUser        // 🔹 info_user dari infotabel
        ]);
    }




    public function saveData(string $table)
    {
        $model = new \App\Models\GenericModel();
        $model->setTable($table);

        $data = $this->request->getPost();
        $id   = $data['id'] ?? null;
        unset($data['id']);

        try {
            if ($id) {
                $model->update($id, $data);
                $msg = 'Data berhasil diperbarui';
            } else {
                $model->insert($data);
                $msg = 'Data berhasil ditambahkan';
            }
            return $this->response->setJSON(['success' => true, 'message' => $msg]);
        } catch (\Throwable $e) {
            return $this->response->setJSON(['success' => false, 'message' => $e->getMessage()]);
        }
    }
    public function getData(string $table, $id)
    {
        $model = new \App\Models\GenericModel();
        $model->setTable($table);
        $data = $model->find($id);

        if ($data)
            return $this->response->setJSON(['success' => true, 'data' => $data]);
        else
            return $this->response->setJSON(['success' => false, 'message' => 'Data tidak ditemukan']);
    }

    public function get(string $table, $id)
    {
        $model = new \App\Models\GenericModel();
        $meta = $model->getTableMeta($table);
        $primaryKey = null;

        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
                break;
            }
        }

        if (!$primaryKey) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Primary key tidak ditemukan.']);
        }

        $row = $model->db->table($table)->where($primaryKey, $id)->get()->getRowArray();

        if ($row) {
            return $this->response->setJSON(['status' => 'success', 'data' => $row]);
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Data tidak ditemukan.']);
        }
    }




    public function listData(string $table)
    {
        $model = $this->newModel($table);
        $rows = $model->findAll();
        return $this->response->setJSON(['data' => $rows]);
    }


    public function form(string $table, $id = null)
    {
        $model = new \App\Models\GenericModel();
        $data['tables'] = $model->getAllTables();
        $data['table']  = $table;
        $data['row']    = $id ? $model->setTable($table)->find($id) : null;

        echo view('layouts/main', $data + ['page' => 'crud/form']);
    }

    public function getRow(string $table, int $id)
    {
        $model = new \App\Models\GenericModel();
        $meta = $model->getTableMeta($table);
        $primaryKey = null;

        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
                break;
            }
        }

        if (!$primaryKey) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Primary key tidak ditemukan']);
        }

        $row = $model->db->table($table)->where($primaryKey, $id)->get()->getRowArray();

        return $this->response->setJSON(['status' => 'success', 'data' => $row]);
    }

    public function save(string $table)
    {
        $model = new \App\Models\GenericModel();
        $data  = $this->request->getPost();

        $meta = $model->getTableMeta($table);
        $primaryKey = null;
        $cleanData = [];

        // 🔍 Cari primary key
        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
            }

            // ✅ Hanya ambil kolom yang valid dan ada di $data
            $colName = $col['name'];
            if (array_key_exists($colName, $data)) {
                $value = $data[$colName];

                // Normalisasi tipe data dasar agar tidak error
                if ($col['type'] === 'int' || $col['type'] === 'tinyint' || $col['type'] === 'bigint') {
                    $value = ($value === '' ? null : (int) $value);
                } elseif ($col['type'] === 'decimal' || $col['type'] === 'float' || $col['type'] === 'double') {
                    $value = ($value === '' ? null : (float) $value);
                } elseif ($col['type'] === 'date' && $value === '') {
                    $value = null;
                } elseif ($col['type'] === 'enum' || $col['type'] === 'set') {
                    // enum/set tetap string, tapi pastikan valid nilainya
                    $allowed = $this->extractEnumValues($col['type_raw']);
                    if (!in_array($value, $allowed)) $value = null;
                }

                $cleanData[$colName] = $value;
            }
        }

        if (!$primaryKey) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Primary key tidak ditemukan pada tabel.'
            ]);
        }

        $builder = $model->db->table($table);
        $lastId = null;

        if (!empty($data[$primaryKey])) {
            // === UPDATE ===
            $id = $data[$primaryKey];
            unset($cleanData[$primaryKey]);
            $builder->where($primaryKey, $id)->update($cleanData);
            $lastId = $id;
        } else {
            // === INSERT BARU ===
            unset($cleanData[$primaryKey]);
            $builder->insert($cleanData);
            $lastId = $model->db->insertID();
        }

        return $this->response->setJSON([
            'status' => 'success',
            'message' => 'Data berhasil disimpan',
            'last_id' => $lastId,
        ]);
    }

    /**
     * 🔍 Ekstrak nilai enum/set dari definisi kolom
     * Contoh: enum('Aktif','Tidak Aktif') → ['Aktif','Tidak Aktif']
     */
    private function extractEnumValues(string $typeRaw): array
    {
        if (preg_match("/'(.*?)'/", $typeRaw)) {
            preg_match_all("/'([^']+)'/", $typeRaw, $matches);
            return $matches[1] ?? [];
        }
        return [];
    }


    public function delete(string $table, int $id)
    {
        $model = new \App\Models\GenericModel();

        // ambil primary key dari meta
        $meta = $model->getTableMeta($table);
        $primaryKey = null;

        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
                break;
            }
        }

        if (!$primaryKey) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Primary key tidak ditemukan.']);
        }

        $builder = $model->db->table($table);
        $builder->where($primaryKey, $id)->delete();

        if ($model->db->affectedRows() > 0) {
            return $this->response->setJSON(['status' => 'success', 'message' => 'Data berhasil dihapus.']);
        } else {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Data gagal dihapus.']);
        }
    }
    public function deleteAll(string $table)
    {
        $model = new \App\Models\GenericModel();
        $db = $model->db;

        // Daftar tabel yang TIDAK boleh dihapus semua datanya
        // $protectedTables = ['user', 'table_settings'];
        $protectedTables = ['user'];

        // Cegah jika tabel masuk daftar terlarang
        if (in_array($table, $protectedTables)) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Tabel ini dilindungi dan tidak dapat dihapus seluruh datanya.'
            ]);
        }

        // Pastikan tabel benar-benar ada di database
        if (!$db->tableExists($table)) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Tabel tidak ditemukan.'
            ]);
        }

        try {
            // Jalankan TRUNCATE untuk hapus semua data dan reset auto_increment
            $db->table($table)->truncate();

            return $this->response->setJSON([
                'status' => 'success',
                'message' => 'Semua data di tabel "' . $table . '" berhasil dihapus.'
            ]);
        } catch (\Throwable $e) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Gagal menghapus semua data: ' . $e->getMessage()
            ]);
        }
    }


    public function export(string $table)
    {
        $model = new \App\Models\GenericModel();
        $fields = $model->getTableFields($table);
        $rows   = $model->getTableData($table);

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom
        $col = 1;
        foreach ($fields as $field) {
            $columnLetter = Coordinate::stringFromColumnIndex($col);
            $sheet->setCellValue($columnLetter . '1', ucfirst($field));
            $col++;
        }

        // Isi data
        $rowIndex = 2;
        foreach ($rows as $row) {
            $col = 1;
            foreach ($fields as $field) {
                $columnLetter = Coordinate::stringFromColumnIndex($col);
                $sheet->setCellValue($columnLetter . $rowIndex, $row[$field] ?? '');
                $col++;
            }
            $rowIndex++;
        }

        $filename = $table . '_export_' . date('Ymd_His') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer = new Xlsx($spreadsheet);
        $writer->save('php://output');
        exit;
    }

    public function import(string $table)
    {
        $file = $this->request->getFile('file');
        if (!$file->isValid()) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'File tidak valid.'
            ]);
        }

        $ext = $file->getClientExtension();
        if (!in_array($ext, ['xls', 'xlsx'])) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Format file harus .xls atau .xlsx'
            ]);
        }

        $model  = new \App\Models\GenericModel();
        $fields = $model->getTableFields($table);
        $meta   = $model->getTableMeta($table);

        // Cari primary key
        $primaryKey = null;
        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
                break;
            }
        }

        $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file->getTempName());
        $sheetData   = $spreadsheet->getActiveSheet()->toArray(null, true, true, true);

        // Buang header (baris pertama)
        array_shift($sheetData);

        $db       = \Config\Database::connect();
        $builder  = $db->table($table);
        $inserted = 0;
        $updated  = 0;
        $insertedIds = []; // ✅ inisialisasi agar tidak undefined

        foreach ($sheetData as $row) {
            $data = [];
            $i = 0;
            foreach ($fields as $f) {
                $i++;
                // Kolom Excel: A=65 (chr(65)), B=66, dst.
                $data[$f] = $row[chr(64 + $i)] ?? null;
            }

            if ($primaryKey) {
                $id = $data[$primaryKey] ?? null;

                if ($id) {
                    // Cek apakah sudah ada
                    $exists = $builder->where($primaryKey, $id)->countAllResults() > 0;

                    if ($exists) {
                        // Update
                        $builder->where($primaryKey, $id)->update($data);
                        $updated++;
                    } else {
                        // Insert baru
                        $builder->insert($data);
                        $inserted++;
                        $insertedIds[] = $db->insertID();
                    }
                } else {
                    // Tidak ada id (data baru)
                    unset($data[$primaryKey]);
                    $builder->insert($data);
                    $inserted++;
                    $insertedIds[] = $db->insertID();
                }
            } else {
                // Tabel tanpa primary key
                $builder->insert($data);
                $inserted++;
                $insertedIds[] = $db->insertID();
            }
        }

        $message = "Import selesai. $inserted data baru ditambahkan, $updated data diperbarui.";

        return $this->response->setJSON([
            'status'  => 'success',
            'message' => $message,
            'inserted_ids' => $insertedIds,
            'inserted' => $inserted,
            'updated'  => $updated,
        ]);
    }
    public function structure(string $table)
    {
        $model = new \App\Models\GenericModel();
        $meta = $model->getTableMeta($table);

        return $this->response->setJSON([
            'status' => 'success',
            'meta' => $meta
        ]);
    }

    private function getTableStructure($table)
    {
        $fields = $this->db->getFieldData($table);
        $columns = [];

        foreach ($fields as $f) {
            $col = [
                'name' => $f->name,
                'type' => $f->type,
                'max_length' => $f->max_length,
                'default' => $f->default,
            ];

            // ✅ deteksi ENUM/SET dari definisi field (pakai query SHOW COLUMNS)
            $query = $this->db->query("SHOW COLUMNS FROM `$table` LIKE '{$f->name}'")->getRow();
            if ($query && preg_match("/^(enum|set)\((.+)\)$/i", $query->Type, $matches)) {
                $col['enumValues'] = array_map(function ($v) {
                    return trim($v, " '");
                }, explode(',', $matches[2]));
            }

            $columns[] = $col;
        }

        return $columns;
    }

    //toggle status
    public function updateToggle()
    {
        $request = service('request');
        $table  = $request->getPost('table');
        $column = $request->getPost('column');
        $id     = $request->getPost('id');
        $status = $request->getPost('status');

        if (!$table || !$column || !$id) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Data tidak lengkap']);
        }

        $model = new \App\Models\GenericModel();

        try {
            // Set table sebelum update
            $db = \Config\Database::connect();
            $builder = $db->table($table);

            $builder->where($builder->primaryKey ?? 'id', $id)
                ->update([$column => $status]);

            return $this->response->setJSON(['status' => 'success', 'message' => 'Berhasil update']);
        } catch (\Exception $e) {
            return $this->response->setJSON(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }
}
