<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use App\Models\GenericModel;
use Config\Database;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;

class CrudController extends Controller
{
    protected $helpers = ['url', 'form', 'filesystem'];
    protected $db;

    public function __construct()
    {
        $this->db = Database::connect();
    }

    protected function newModel(string $table): GenericModel
    {
        $model = new GenericModel();
        $model->setTable($table);
        return $model;
    }

    // ===================================================================
    // INDEX: Halaman utama CRUD
    // ===================================================================
    public function index($table = null)
    {
        // Ambil instance session CodeIgniter
        $session = session();

        // Panggil model generik yang bisa digunakan untuk berbagai tabel
        $model   = new GenericModel();

        // Koneksi ke database secara manual
        $db      = Database::connect();

        // Ambil role user dari session (misal: admin, user, dll)
        $userRole    = $session->get('user_role');

        // Ambil tabel terakhir yang dikunjungi di sidebar
        $lastTable   = $session->get('lastSidebarTable');

        // Jika parameter $table kosong, gunakan tabel terakhir yang disimpan di session
        if (!empty($lastTable) && empty($table)) {
            $table = $lastTable;
        }

        // ================================================================
        // Ambil daftar tabel yang diizinkan untuk role user saat ini
        // ================================================================
        $query = $db->table('table_settings')
            ->select('table_name, icon')        // Ambil nama tabel dan ikon-nya
            ->where('role', $userRole)          // Filter berdasarkan role user
            ->where('menu', 'aktif')            // Hanya tabel dengan menu aktif
            ->get()
            ->getResultArray();                 // Ambil hasil dalam bentuk array

        // Siapkan array kosong untuk tabel dan ikon
        $tables = [];
        $icons  = [];

        // Isi array berdasarkan hasil query
        foreach ($query as $row) {
            $tables[] = $row['table_name'];                            // Simpan nama tabel
            $icons[strtolower($row['table_name'])] = $row['icon'];     // Simpan ikon berdasarkan nama tabel (lowercase)
        }

        // ================================================================
        // Ambil data urutan dan kategori dari tabel 'menu'
        // ================================================================
        $menuQuery = $db->table('menu')
            ->select('table_name, kategori, nomor, menu_order')
            ->whereIn('table_name', $tables)   // Hanya untuk tabel yang diizinkan
            ->orderBy('nomor', 'ASC')          // Urutkan berdasarkan kolom 'nomor'
            ->orderBy('menu_order', 'ASC')     // Lalu berdasarkan 'menu_order'
            ->get()
            ->getResultArray();

        // ================================================================
        // Kelompokkan tabel berdasarkan kategori (misal: Master Data, Transaksi)
        // ================================================================
        $groupedTables = [];
        foreach ($menuQuery as $row) {
            // Jika kategori kosong, beri label 'Tanpa Kategori'
            $kategori = $row['kategori'] ?: 'Tanpa Kategori';

            // Masukkan tabel ke dalam kelompok kategori
            $groupedTables[$kategori][] = $row['table_name'];
        }

        // ================================================================
        // Deteksi apakah user sedang berada dalam "Active View" mode
        // ================================================================
        $activeTable     = $session->get('activeViewTable');  // Ambil tabel aktif yang disimpan di session
        $isActiveView    = !empty($activeTable);              // True jika sedang ada active view

        // ================================================================
        // Siapkan semua data untuk dikirim ke view
        // ================================================================
        $data = [
            'tables'          => $tables,                             // Daftar tabel yang diizinkan
            'icons'           => $icons,                              // Ikon tiap tabel
            'groupedTables'   => $groupedTables,                      // Tabel yang dikelompokkan per kategori
            'table'           => $table ?? null,                      // Nama tabel aktif saat ini (jika ada)
            'fields'          => $table ? $model->getTableFields($table) : [],  // Ambil field kolom tabel
            'meta'            => $table ? $model->getTableMeta($table) : [],    // Ambil metadata tabel
            'user'            => $session->get(),                     // Data session user
            'isActiveView'    => $isActiveView,                       // Status apakah dalam active view
            'activeViewTable' => $activeTable,                        // Nama tabel active view
        ];

        // Jika user tidak punya akses ke tabel apa pun
        if (empty($tables)) {
            $data['noAccess'] = 'Tidak ada tabel yang diizinkan untuk role ini.';
        }

        // ================================================================
        // Render halaman utama CRUD dengan data yang sudah disiapkan
        // ================================================================
        return view('crud/content', $data);
    }

    // ===================================================================
    // LOAD TABLE CONTENT: Pilih mode ActiveView atau Normal
    // ===================================================================
    public function loadTableContent(string $table)
    {
        $session     = session();
        $activeTable = $session->get('activeViewTable');

        return !empty($activeTable)
            ? $this->loadActiveView($table)
            : $this->loadNormal($table);
    }

    // ===================================================================
    // ACTIVE VIEW: Tampilan khusus dari tabel konfigurasi
    // ===================================================================
    protected function loadActiveView(string $table)
    {
        $model   = new GenericModel();
        $db      = Database::connect();
        $session = session();

        $userRole        = $session->get('user_role') ?? $session->get('role');
        $sessionTable    = $session->get('activeViewTable');
        $lastSidebarTable = $session->get('lastSidebarTable');

        if (empty($sessionTable)) {
            return 'Tabel aktif tidak ditemukan di session.';
        }

        $fields = $model->getTableFields($sessionTable);
        $meta   = $model->getTableMeta($sessionTable);

        $rows = $db->tableExists($sessionTable)
            ? $db->table($sessionTable)->where('table_name', $lastSidebarTable)->get()->getResultArray()
            : [];

        // Tombol aksi dari table_settings
        $buttons = array_fill_keys(
            ['tambah', 'print', 'export', 'import', 'view', 'edit', 'delete', 'generate', 'delete_all', 'activeView'],
            'nonaktif'
        );

        if ($userRole) {
            $query = $db->table('table_settings')
                ->select('tambah, print, export, import, view, edit, delete, generate, delete_all, activeView')
                ->where('role', $userRole)
                ->where('table_name', $sessionTable)
                ->get()
                ->getRowArray();

            if ($query) {
                foreach ($buttons as $k => $v) {
                    if (!empty($query[$k]) && $query[$k] === 'aktif') {
                        $buttons[$k] = 'aktif';
                    }
                }
            }
        }

        // Field visible dari field_settings
        $visibleFields = [];
        if ($userRole) {
            $visibleFields = array_column(
                $db->table('field_settings')
                    ->select('field_name')
                    ->where('table_name', $sessionTable)
                    ->where('role', $userRole)
                    ->where('visible', 'aktif')
                    ->get()
                    ->getResultArray(),
                'field_name'
            );
        }
        if (empty($visibleFields)) {
            $visibleFields = $fields;
        }

        // Toggle fields
        $toggleFields = [];
        if ($userRole) {
            $toggleFields = array_column(
                $db->table('toggle_fields')
                    ->select('field_name')
                    ->where('table_name', $sessionTable)
                    ->where('role', $userRole)
                    ->where('toggle_on', 'aktif')
                    ->get()
                    ->getResultArray(),
                'field_name'
            );
        }

        // Icon tabel
        $icon = 'fa-database';
        if ($userRole) {
            $iconQuery = $db->table('table_settings')
                ->select('icon')
                ->where('role', $userRole)
                ->where('table_name', $sessionTable)
                ->get()
                ->getRowArray();

            if (!empty($iconQuery['icon'])) {
                $icon = $iconQuery['icon'];
            }
        }

        // Info tambahan dari infotabel
        $infoTabel = $infoUser = '';
        if ($userRole) {
            $infoQuery = $db->table('infotabel')
                ->select('info_tabel, info_user')
                ->where('nama_tabel', $sessionTable)
                ->where('role', $userRole)
                ->get()
                ->getRowArray();

            if ($infoQuery) {
                $infoTabel = $infoQuery['info_tabel'] ?? '';
                $infoUser  = $infoQuery['info_user'] ?? '';
            }
        }

        return view('crud/activeView', [
            'table'           => $lastSidebarTable,
            'activeViewTable' => $sessionTable,
            'fields'          => $visibleFields,
            'meta'            => $meta,
            'rows'            => $rows,
            'buttons'         => $buttons,
            'toggleFields'    => $toggleFields,
            'icon'            => $icon,
            'infoTabel'       => $infoTabel,
            'infoUser'        => $infoUser,
            'isActiveView'    => true,
            'sessionTable'    => $sessionTable,
            'keysortir'       => $sessionTable,
        ]);
    }

    // ===================================================================
    // NORMAL VIEW: Tampilan tabel standar
    // ===================================================================
    protected function loadNormal(string $table)
    {
        $model   = new GenericModel();
        $db      = Database::connect();
        $session = session();

        $userRole = $session->get('user_role') ?? $session->get('role') ?? null;

        $fields = $model->getTableFields($table);
        $meta   = $model->getTableMeta($table);
        $rows   = $model->getTableData($table);

        // Tombol aksi
        $buttonSettings = array_fill_keys(
            ['tambah', 'print', 'export', 'import', 'view', 'edit', 'delete', 'generate', 'delete_all', 'activeView'],
            'nonaktif'
        );

        if ($userRole) {
            $query = $db->table('table_settings')
                ->select('tambah, print, export, import, view, edit, delete, generate, delete_all, activeView')
                ->where('role', $userRole)
                ->where('table_name', $table)
                ->get()
                ->getRowArray();

            if ($query) {
                foreach ($buttonSettings as $k => $v) {
                    if (isset($query[$k]) && $query[$k] === 'aktif') {
                        $buttonSettings[$k] = 'aktif';
                    }
                }
            }
        }

        // Field visible
        $visibleFields = [];
        if ($userRole) {
            $visibleFields = array_column(
                $db->table('field_settings')
                    ->select('field_name')
                    ->where('table_name', $table)
                    ->where('role', $userRole)
                    ->where('visible', 'aktif')
                    ->get()
                    ->getResultArray(),
                'field_name'
            );
        }
        if (empty($visibleFields)) {
            $visibleFields = $fields;
        }

        // Toggle fields
        $toggleFields = [];
        if ($userRole) {
            $toggleFields = array_column(
                $db->table('toggle_fields')
                    ->select('field_name')
                    ->where('table_name', $table)
                    ->where('role', $userRole)
                    ->where('toggle_on', 'aktif')
                    ->get()
                    ->getResultArray(),
                'field_name'
            );
        }

        // Icon & info
        $icon = 'fa-database';
        if ($userRole) {
            $iconQuery = $db->table('table_settings')
                ->select('icon')
                ->where('role', $userRole)
                ->where('table_name', $table)
                ->get()
                ->getRowArray();
            if (!empty($iconQuery['icon'])) {
                $icon = $iconQuery['icon'];
            }
        }

        $infoTabel = $infoUser = '';
        if ($userRole) {
            $infoQuery = $db->table('infotabel')
                ->select('info_tabel, info_user')
                ->where('nama_tabel', $table)
                ->where('role', $userRole)
                ->get()
                ->getRowArray();

            if ($infoQuery) {
                $infoTabel = $infoQuery['info_tabel'] ?? '';
                $infoUser  = $infoQuery['info_user'] ?? '';
            }
        }

        return view('crud/table_content', [
            'table'          => $table,
            'fields'         => $visibleFields,
            'meta'           => $meta,
            'rows'           => $rows,
            'buttons'        => $buttonSettings,
            'toggleFields'   => $toggleFields,
            'icon'           => $icon,
            'infoTabel'      => $infoTabel,
            'infoUser'       => $infoUser,
            'isActiveView'   => false,
            'activeViewTable' => null,
        ]);
    }

    // ===================================================================
    // SAVE DATA (Tambah/Edit)
    // ===================================================================
    public function save(string $table)
    {
        $session = session();
        $table   = $table ?? $this->request->getGet('table');

        $model   = new GenericModel();
        $data    = $this->request->getPost();

        $activeViewTable = $session->get('activeViewTable');
        $isActiveView    = !empty($activeViewTable);
        $targetTable     = $isActiveView ? $activeViewTable : $table;

        $model->setTable($targetTable);
        $meta = $model->getTableMeta($targetTable);

        $primaryKey = null;
        $cleanData  = [];

        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
            }

            $colName = $col['name'];
            if (array_key_exists($colName, $data)) {
                $value = $data[$colName];

                if (in_array($col['type'], ['int', 'tinyint', 'bigint'])) {
                    $value = ($value === '' ? null : (int)$value);
                } elseif (in_array($col['type'], ['decimal', 'float', 'double'])) {
                    $value = ($value === '' ? null : (float)$value);
                } elseif ($col['type'] === 'date' && $value === '') {
                    $value = null;
                } elseif (in_array($col['type'], ['enum', 'set'])) {
                    $allowed = $this->extractEnumValues($col['type_raw']);
                    if (!in_array($value, $allowed)) $value = null;
                }

                $cleanData[$colName] = $value;
            }
        }

        if (!$primaryKey) {
            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Primary key tidak ditemukan pada tabel ' . $targetTable
            ]);
        }

        $builder = $model->db->table($targetTable);
        $lastId  = null;

        // if ($isActiveView && $model->hasField($targetTable, 'table_name')) {
        //     $cleanData['table_name'] = $table;
        // }

        if (!empty($data[$primaryKey])) {
            $id = $data[$primaryKey];
            unset($cleanData[$primaryKey]);
            $builder->where($primaryKey, $id)->update($cleanData);
            $lastId = $id;
        } else {
            unset($cleanData[$primaryKey]);
            $builder->insert($cleanData);
            $lastId = $model->db->insertID();
        }

        $modeLabel = $isActiveView ? "(Active View: $activeViewTable)" : "(Normal)";
        return $this->response->setJSON([
            'status'   => 'success',
            'message'  => "Data berhasil disimpan {$modeLabel}",
            'last_id'  => $lastId,
            'saved_to' => $targetTable
        ]);
    }

    // ===================================================================
    // UTILITIES
    // ===================================================================
    private function extractEnumValues(string $typeRaw): array
    {
        if (preg_match("/'(.*?)'/", $typeRaw)) {
            preg_match_all("/'([^']+)'/", $typeRaw, $matches);
            return $matches[1] ?? [];
        }
        return [];
    }

    // ===================================================================
    // GET DATA
    // ===================================================================
    public function get(string $table, $id)
    {
        $model = new GenericModel();
        $meta  = $model->getTableMeta($table);
        $primaryKey = null;

        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
                break;
            }
        }

        if (!$primaryKey) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Primary key tidak ditemukan.']);
        }

        $row = $model->db->table($table)->where($primaryKey, $id)->get()->getRowArray();

        return $this->response->setJSON(
            $row
                ? ['status' => 'success', 'data' => $row]
                : ['status' => 'error', 'message' => 'Data tidak ditemukan.']
        );
    }

    // ===================================================================
    // DELETE & DELETE ALL
    // ===================================================================
    public function delete(string $table, int $id)
    {
        $model = new GenericModel();
        $meta  = $model->getTableMeta($table);
        $primaryKey = null;

        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
                break;
            }
        }

        if (!$primaryKey) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Primary key tidak ditemukan.']);
        }

        $builder = $model->db->table($table);
        $builder->where($primaryKey, $id)->delete();

        return $this->response->setJSON(
            $model->db->affectedRows() > 0
                ? ['status' => 'success', 'message' => 'Data berhasil dihapus.']
                : ['status' => 'error', 'message' => 'Data gagal dihapus.']
        );
    }

    public function deleteAll(string $table)
    {
        $db = $this->db;
        $protectedTables = ['user'];

        if (in_array($table, $protectedTables)) {
            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Tabel ini dilindungi dan tidak dapat dihapus seluruh datanya.'
            ]);
        }

        if (!$db->tableExists($table)) {
            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Tabel tidak ditemukan.'
            ]);
        }

        try {
            $db->table($table)->truncate();
            return $this->response->setJSON([
                'status'  => 'success',
                'message' => "Semua data di tabel \"{$table}\" berhasil dihapus."
            ]);
        } catch (\Throwable $e) {
            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Gagal menghapus semua data: ' . $e->getMessage()
            ]);
        }
    }

    // ===================================================================
    // EXPORT & IMPORT
    // ===================================================================
    public function export(string $table)
    {
        $model  = new GenericModel();
        $fields = $model->getTableFields($table);
        $rows   = $model->getTableData($table);

        $spreadsheet = new Spreadsheet();
        $sheet       = $spreadsheet->getActiveSheet();

        $col = 1;
        foreach ($fields as $field) {
            $columnLetter = Coordinate::stringFromColumnIndex($col);
            $sheet->setCellValue($columnLetter . '1', ucfirst($field));
            $col++;
        }

        $rowIndex = 2;
        foreach ($rows as $row) {
            $col = 1;
            foreach ($fields as $field) {
                $columnLetter = Coordinate::stringFromColumnIndex($col);
                $sheet->setCellValue($columnLetter . $rowIndex, $row[$field] ?? '');
                $col++;
            }
            $rowIndex++;
        }

        $filename = $table . '_export_' . date('Ymd_His') . '.xlsx';
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer = new Xlsx($spreadsheet);
        $writer->save('php://output');
        exit;
    }

    public function import(string $table)
    {
        $file = $this->request->getFile('file');
        if (!$file->isValid()) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'File tidak valid.']);
        }

        $ext = $file->getClientExtension();
        if (!in_array($ext, ['xls', 'xlsx'])) {
            return $this->response->setJSON(['status' => 'error', 'message' => 'Format file harus .xls atau .xlsx']);
        }

        $model  = new GenericModel();
        $fields = $model->getTableFields($table);
        $meta   = $model->getTableMeta($table);

        $primaryKey = null;
        foreach ($meta as $col) {
            if ($col['key'] === 'PRI') {
                $primaryKey = $col['name'];
                break;
            }
        }

        $spreadsheet = IOFactory::load($file->getTempName());
        $sheetData   = $spreadsheet->getActiveSheet()->toArray(null, true, true, true);
        array_shift($sheetData);

        $db       = Database::connect();
        $builder  = $db->table($table);
        $inserted = $updated = 0;
        $insertedIds = [];

        foreach ($sheetData as $row) {
            $data = [];
            $i = 0;
            foreach ($fields as $f) {
                $i++;
                $data[$f] = $row[chr(64 + $i)] ?? null;
            }

            if ($primaryKey) {
                $id = $data[$primaryKey] ?? null;
                if ($id && $builder->where($primaryKey, $id)->countAllResults() > 0) {
                    $builder->where($primaryKey, $id)->update($data);
                    $updated++;
                } else {
                    unset($data[$primaryKey]);
                    $builder->insert($data);
                    $inserted++;
                    $insertedIds[] = $db->insertID();
                }
            } else {
                $builder->insert($data);
                $inserted++;
                $insertedIds[] = $db->insertID();
            }
        }

        $message = "Import selesai. $inserted data baru ditambahkan, $updated data diperbarui.";
        return $this->response->setJSON([
            'status'       => 'success',
            'message'      => $message,
            'inserted_ids' => $insertedIds,
            'inserted'     => $inserted,
            'updated'      => $updated,
        ]);
    }

    // ===================================================================
    // STRUCTURE & TOGGLE
    // ===================================================================
    public function structure(string $table)
    {
        $model = new GenericModel();
        $meta  = $model->getTableMeta($table);

        return $this->response->setJSON(['status' => 'success', 'meta' => $meta]);
    }

    public function updateToggle()
    {
        $request = service('request');
        $table   = $request->getPost('table');
        $column  = $request->getPost('column');
        $id      = $request->getPost('id');
        $status  = $request->getPost('status'); // 'aktif' atau 'nonaktif'

        if (!$table || !$column || !$id) {
            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Data tidak lengkap untuk update toggle.'
            ]);
        }

        $db = Database::connect();
        $builder = $db->table($table);

        try {
            // =====================================================
            // 🧠 NORMALISASI ID (contoh: "switch_active_199" → 199)
            // =====================================================
            // Ambil hanya angka terakhir setelah underscore
            if (preg_match('/([0-9]+)$/', $id, $matches)) {
                $id = $matches[1];
            }

            // =====================================================
            // 🔍 CARI PRIMARY KEY OTOMATIS
            // =====================================================
            $fields = $db->getFieldData($table);
            $primaryKey = 'id';
            foreach ($fields as $f) {
                if (!empty($f->primary_key)) {
                    $primaryKey = $f->name;
                    break;
                }
            }

            // =====================================================
            // ✅ PASTIKAN RECORD ADA
            // =====================================================
            $exists = $builder->where($primaryKey, $id)->countAllResults(false);
            if ($exists == 0) {
                return $this->response->setJSON([
                    'status'  => 'error',
                    'message' => "Data dengan {$primaryKey} = {$id} tidak ditemukan di tabel {$table}."
                ]);
            }

            // =====================================================
            // 🚀 UPDATE STATUS
            // =====================================================
            $builder->where($primaryKey, $id)->update([$column => $status]);

            if ($db->affectedRows() > 0) {
                return $this->response->setJSON([
                    'status'  => 'success',
                    'message' => "Berhasil ubah status '{$column}' menjadi '{$status}' pada ID {$id} di tabel {$table}.",
                    'updated' => [
                        'table'  => $table,
                        'column' => $column,
                        'id'     => $id,
                        'value'  => $status
                    ]
                ]);
            } else {
                return $this->response->setJSON([
                    'status'  => 'warning',
                    'message' => "Tidak ada data yang berubah (kemungkinan status sudah '{$status}')."
                ]);
            }
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Gagal update toggle: ' . $e->getMessage()
            ]);
        }
    }


    // ===================================================================
    // ACTIVE VIEW SESSION
    // ===================================================================
    public function activeView(string $table)
    {
        $session    = session();
        $sessionKey = 'activeViewTable';
        $current    = $session->get($sessionKey);

        if ($current === $table) {
            $session->remove($sessionKey);
            return $this->response->setJSON([
                'success' => true,
                'active'  => false,
                'message' => "Active view untuk tabel '$table' telah dinonaktifkan."
            ]);
        }

        $session->set($sessionKey, $table);
        $session->set('isActiveViewMode', true);

        return $this->response->setJSON([
            'success' => true,
            'active'  => true,
            'message' => "Active view diatur ke tabel '$table'."
        ]);
    }

    public function clearActiveView()
    {
        $session = session();
        $session->remove('activeViewTable');
        return $this->response->setJSON(['success' => true, 'message' => 'Session activeViewTable telah dihapus.']);
    }

    public function saveLastTable()
    {
        $session = session();
        $table   = $this->request->getPost('table');

        if ($table) {
            $session->set('lastSidebarTable', $table);
            return $this->response->setJSON(['success' => true]);
        }

        return $this->response->setJSON(['success' => false, 'message' => 'Table kosong']);
    }

    public function setActiveViewSession()
    {
        $session = session();
        $table   = $this->request->getPost('table');

        if (!empty($table)) {
            $session->set('activeViewTable', $table);
            return $this->response->setJSON(['success' => true, 'message' => "ActiveView diset ke {$table}"]);
        }

        $session->remove('activeViewTable');
        return $this->response->setJSON(['success' => true, 'message' => 'ActiveView dihapus']);
    }

    // public function setLastSidebarTableSession()
    // {
    //     $table = $this->request->getPost('table');
    //     session()->set('lastSidebarTable', $table);

    //     return $this->response->setJSON([
    //         'success' => true,
    //         'stored' => $table
    //     ]);
    // }
}
